
#include "flashprg.h"        /* Header file with global prototypes */

/* reloctable function calling in C! */

void FlashReadReset( void );
void FlashPause( unsigned int uMicroSeconds );
int FlashBlockErase( unsigned int ucNumBlocks, unsigned char ucBlock[],unsigned long *BlockOffset );
int FlashProgram( unsigned long ulOff, size_t NumWords, void *Array );
int FlashDataPoll( unsigned long ulOff, unsigned int icVal );

/* No RAM data left!!! RAM data may overlap with the RAM image! . */

/*******************************************************************************
Function:    unsigned char FlashWrite( unsigned long ulOff, unsigned char ucVal)
Arguments:   ulOff is byte offset into the flash to write to.
   ucVal is the value to be written
Returns:     ucVal
Description: This function is used to write a value to the flash. On many
   microprocessor systems a macro can be used instead, increasing the speed of
   the flash routines. For example:
*/

#define FlashWrite( ulOff, ucVal) ( BASE_ADDR[ulOff] = (SIZEOFADDRESSEDFLASHUNIT) ucVal )

/*                              
   A function is used here instead to allow the user to expand it if necessary.
   The function is made to return ucVal so that it is compatible with the macro.

Pseudo Code:
   Step 1: Write ucVal to the byte offset in the flash
   Step 2: return ucVal

******************************************************************************
static unsigned char FlashWrite( unsigned long ulOff, unsigned char ucVal )
{
   return BASE_ADDR[ulOff] = ucVal;
}
*/
/*******************************************************************************
Function:    unsigned char FlashRead( unsigned long ulOff )
Arguments:   ulOff is byte offset into the flash to read from.
Returns:     The unsigned char at the byte offset
Description: This function is used to read a byte from the flash. On many
   microprocessor systems a macro can be used instead, increasing the speed of
   the flash routines. For example:
*/
#define FlashRead( ulOff ) ( BASE_ADDR[ulOff] )
/*
  A function is used here instead to allow the user to expand it if necessary.

Pseudo Code:
  Step 1: Return the value at byte offset ulOff

******************************************************************************
unsigned short FlashRead( unsigned long ulOff )
{
   return BASE_ADDR[ulOff];
}
*/

/*******************************************************************************
Function:      void FlashReadReset( void )
Arguments:     none
Return Value:  none
Description:   This function places the flash in the Read Array mode described
   in the Data Sheet. In this mode the flash can be read as normal memory.

   All of the other functions leave the flash in the Read Array mode so this is
   not strictly necessary. It is provided for completeness.

Note: A wait of 10us is required if the command is called during a program or
   erase instruction. This is included here to guarantee operation. The
   functions in the data sheet call this function if they suspect an error
   during programming or erasing so that the 10us pause is included. Otherwise
   they use the single instruction technique for increased speed.

Pseudo Code:
   Step 1: write command sequence (see Instructions Table of the Data Sheet)
   Step 2: wait 10us

*******************************************************************************/
void FlashReadReset( void )
{
   /* Step 1: write command sequence */
   FlashWrite( FLASH_OFFSET1, FLASH_PATTERN_CYCLE1 );  /* 1st Cycle */
   FlashWrite( FLASH_OFFSET2, FLASH_PATTERN_CYCLE2 );  /* 2nd Cycle */
   FlashWrite( FLASH_OFFSET1, FLASH_PATTERN_ENTER_READ_MODE );  /* 3rd Cycle */

   /* Step 2: wait 10us. This is a call to FlashPause. */
   FlashPause(10);
}

/*******************************************************************************
Function:    void FlashPause( unsigned int uMicroSeconds )
Arguments:   uMicroSeconds: length of the pause in microseconds
Returns:     none
Description: This routine returns after uMicroSeconds have elapsed. It is used
   in several parts of the code to generate a pause required for correct
   operation of the flash part.

   The routine here works by counting. The user may already have a more suitable
   routine for timing which can be used.

Pseudo Code:
   Step 1: Compute count size for pause.
   Step 2: Count to the required size.

*****************************************************************/
void FlashPause( unsigned int uMicroSeconds )
{
   volatile unsigned long ulCountSize;

   /* Step 1: Compute the count size */
   ulCountSize = (unsigned long)uMicroSeconds * COUNTS_PER_MICROSECOND;

   /* Step 2: Count to the required size */
   while( ulCountSize > 0 )   /* Test to see if finished */
      ulCountSize--;          /* and count down */
}

/*******************************************************************************
Function:     int FlashBlockErase( unsigned char ucNumBlocks,
   unsigned char ucBlock[]  )
Arguments:    ucNumBlocks holds the number of blocks in the array ucBlock
   ucBlock is an array containing the blocks to be erased.
Return Value: The function returns the following conditions:
   FLASH_SUCCESS              (-1)
   FLASH_POLL_FAIL            (-2)
   FLASH_TOO_MANY_BLOCKS      (-3)
   FLASH_MPU_TOO_SLOW         (-4)
   FLASH_WRONG_TYPE           (-8)
   Number of the first protected or invalid block

      The user's array, ucBlock[] is used to report errors on the specified
   blocks. If a time-out occurs because the MPU is too slow then the blocks
   in ucBlocks which are not erased are overwritten with FLASH_BLOCK_NOT_ERASED
   (FFh) and the function returns FLASH_MPU_TOO_SLOW.
      If an error occurs during the erasing of the blocks the blocks in
   ucBlocks which have failed the erase are set to FLASH_BLOCK_ERASE_FAILURE
   (FEh) and the function returns FLASH_POLL_FAIL.
      If both errors occur then the function will set the ucBlock array for
   each type of error (i.e. either to FLASH_BLOCK_NOT_ERASED or to
   FLASH_BLOCK_ERASE_FAILURE). It will return FLASH_POLL_FAIL even though
   the FLASH_MPU_TOO_SLOW has also occured.

Description:  This function erases up to ucNumBlocks in the flash. The blocks
  can be listed in any order. The function does not return until the blocks are
  erased. If any blocks are protected or invalid none of the blocks are erased.

  During the Erase Cycle the Data Polling Flowchart of the Data Sheet is
  followed. The toggle bit, DQ6, is not used. For an erase cycle the data on DQ7
  will be '0' during the erase and '1' on completion.

Pseudo Code:
   Step 1:  Check for correct flash type
   Step 2:  Check for protected or invalid blocks
   Step 3:  Write Block Erase command
   Step 4:  Check for time-out blocks
   Step 5:  Wait for the timer bit to be set
   Step 6:  Perform Data Polling until P/E.C. has completed
   Step 7:  Return to Read Array mode
*******************************************************************************/
int FlashBlockErase( unsigned int icNumBlocks, unsigned char ucBlock[],unsigned long *BlockOffset )
{
   unsigned char ucCurBlock;    /* Range Variable to track current block */
   int iRetVal = FLASH_SUCCESS; /* Holds return value: optimistic initially! */
   unsigned char ucNumBlocks=(unsigned char)icNumBlocks;
   unsigned long iCurrent;
   FlashWrite( FLASH_OFFSET1, FLASH_PATTERN_CYCLE1 );
   FlashWrite( FLASH_OFFSET2, FLASH_PATTERN_CYCLE2 );
   FlashWrite( FLASH_OFFSET1, FLASH_PATTERN_WRITEBLOCKERASE );
   FlashWrite( FLASH_OFFSET1, FLASH_PATTERN_CYCLE1 );
   FlashWrite( FLASH_OFFSET2, FLASH_PATTERN_CYCLE2 );
   /* DSI!: Time critical section. Additional blocks must be added every 80us */
   for( ucCurBlock = 0; ucCurBlock < ucNumBlocks; ucCurBlock++ )
   {
      FlashWrite( BlockOffset[ucBlock[ucCurBlock]], FLASH_PATTERN_ERASE_ADD_BLOCK );
      /* Check for Erase Timeout Period */
#ifdef DONTUSETIMEOUT
      if( (FlashRead( BlockOffset[ucBlock[0]] ) & 0x08) == 0x08 )
         break; /* Cannot set any more sectors due to timeout */
#endif
   }
   /* ENI! */

   /* Step 4: Check for time-out blocks */
   /* if timeout occured then check if last block is erasing or not */
   /* Use DQ2 of status register, toggle implies block is erasing */
   if( ucCurBlock < ucNumBlocks )
   {
      iRetVal = FLASH_MPU_TOO_SLOW;
      /* Now specify all other blocks as not being erased */
      while( ucCurBlock < ucNumBlocks )
      {
         ucBlock[ucCurBlock++] = FLASH_BLOCK_NOT_ERASED;
      }
   }

   /* Step 5: Wait for the timer bit to be set */
   while( 1 )  /* TimeOut!: If, for some reason, the hardware fails then this
               loop may not exit. Use a timer function to implement a timeout
               from the loop. */
   {
      if( ( FlashRead( BlockOffset[ucBlock[0]] ) & BIT3MASK ) == BIT3MASK )
         break; /* Break when device starts the erase cycle */
   }
   /* Step 6: Perform data polling until P/E.C. completes, check for errors */
   iCurrent=BlockOffset[ucBlock[0]];
   if( FlashDataPoll(iCurrent,FLASH_PATTERN_ERASED) != FLASH_SUCCESS) 
   {
      iRetVal = FLASH_POLL_FAIL;
   }
   /* Step 7: Return to Read Array mode */
   FlashWrite( 0x0000L, FLASH_PATTERN_ENTER_READ_MODE ); /* Use single instruction cycle method */

   return iRetVal;
}


/*******************************************************************************
Function:     int FlashProgram( unsigned long ulOff, size_t NumWords,
   void *Array )
Arguments:    ulOff is the byte offset into the flash to be programmed
   NumWords holds the number of words in the array.
   Array is a pointer to the array to be programmed.
Return Value: On success the function returns FLASH_SUCCESS (-1).
   On failure the function returns FLASH_PROGRAM_FAIL (-6).
   If the address exceeds the address range of the Flash Device the function
   returns FLASH_ADDRESS_OUT_OF_RANGE (-7) and nothing is programmed.
   If the wrong type of flash is accessed then the function returns
   FLASH_WRONG_TYPE (-8).
Description: This function is used to program an array into the flash. It does
   not erase the flash first and will fail if the block is not erased first.

Pseudo Code:
   Step 1: Check that the flash is of the correct type
   Step 2: Check the offset range is valid.
   Step 3: While there is more to be programmed
   Step 4:  Program the next byte
   Step 5:  Perform data polling until P/E.C. has completed.
   Step 6:  Update pointers
   Step 7: End of While Loop
   Step 8: Return to Read Array mode
*******************************************************************************/

int FlashProgram( unsigned long ulOff, size_t NumWords, void *Array )
{
   SIZEOFADDRESSEDFLASHUNIT *ucArrayPointer; 
   unsigned long ulLastOff;      /* Holds the last offset to be programmed */
   unsigned int iCurrent;
   /* Step 2: Check the address range is valid */
   ulLastOff = ulOff + NumWords - 1;    /* Wegen Sizepruefung */
   /* Step 3: While there is more to be programmed */
   ucArrayPointer = (SIZEOFADDRESSEDFLASHUNIT *)Array;
   while( ulOff <= ulLastOff )
   {
      /* Step 4: Program the next byte */
      FlashWrite( FLASH_OFFSET1, FLASH_PATTERN_CYCLE1 );  /* 1st cycle */
      FlashWrite( FLASH_OFFSET2, FLASH_PATTERN_CYCLE2 );  /* 2nd cycle */
      FlashWrite( FLASH_OFFSET1, FLASH_PATTERN_PROGRAM );  /* Program command */
      FlashWrite( ulOff, *ucArrayPointer );   /* Program value */

      /* Step 5: Perform data polling until P/E.C. has completed. */
      /* See Data Polling Flowchart of the Data Sheet */
      iCurrent=(unsigned int)*ucArrayPointer;
      if( FlashDataPoll(ulOff,iCurrent)== FLASH_POLL_FAIL)
      {
         FlashReadReset();
         return FLASH_PROGRAM_FAIL;
      }

      /* Step 6: Update pointers */
      ulOff++;          /* next byte offset */
      ucArrayPointer++;  /* next word in array */

   /* Step 7: End while loop */
   }

   /* Step 8: Return to Read Array mode */
   FlashWrite( 0x0000L, FLASH_PATTERN_ENTER_READ_MODE ); /* Use single instruction cycle method */
   return FLASH_SUCCESS;
}

/*******************************************************************************
Function:     static int FlashDataPoll( unsigned long ulOff,
                                        unsigned char ucVal )
Arguments:    ulOff should hold a valid offset to be polled. For programming
   this will be the offset of the byte being programmed. For erasing this can
   be any address in the block(s) being erased.
   ucVal should hold the value being programmed. A value of FFh should be used
   when erasing.
Return Value: The function returns FLASH_SUCCESS if the P/E.C. is successful
   or FLASH_POLL_FAIL if there is a problem.
Description: The function is used to monitor the P/E.C. during erase or
   program operations. It returns when the P/E.C. has completed. The Data Sheet
   gives a flow chart (Data Polling Flowchart) showing the operation of the
   function.

Pseudo Code:
   Step 1: Read DQ5 and DQ7 (into a byte)
   Step 2: If DQ7 is the same as Value(bit 7) then return FLASH_SUCCESS
   Step 3: Else if DQ5 is zero then operation is not yet complete, goto 1
   Step 4: Else (DQ5 == 1), Read DQ7
   Step 5: If DQ7 is now the same as Value(bit 7) then return FLASH_SUCCESS
   Step 6: Else return FLASH_POLL_FAIL
*******************************************************************************/
int FlashDataPoll( unsigned long ulOff, unsigned int icVal )
{
   SIZEOFADDRESSEDFLASHUNIT uc;                 /* holds value read from valid offset */
   unsigned short ucVal=(SIZEOFADDRESSEDFLASHUNIT)icVal;
   while( 1 )  /* TimeOut!: If, for some reason, the hardware fails then this
                  loop may not exit. Use a timer function to implement a timeout
                  from the loop. */
   {
      /* Step 1: Read DQ5 and DQ7 (into a byte) */
      uc = FlashRead( ulOff );               /* Read DQ5, DQ7 at valid addr */

   /* Step 2: If DQ7 is the same as Value(bit 7) then return FLASH_SUCCESS */
      if( (uc&BIT7MASK) == (icVal&BIT7MASK) )        /* DQ7 == DATA  */
         return FLASH_SUCCESS;

      /* Step 3: Else if DQ5 is zero then operation is not yet complete */
      /* must do it like this because it may happen that one of the two chips 
         may still be busy whereas the other one is done (happens eg if one of
         the two words to be written is 0xffff */
      if( (uc&BIT5MASK) != BIT5MASK )                /* DQ5 == 0 (1 for Erase Error) */
         continue;

      /* Step 4: Else (DQ5 == 1) */
      uc = FlashRead( ulOff );               /* Read DQ7 at valid addr */

      /* Step 5: If DQ7 is now the same as Value(bit 7) then
         return FLASH_SUCCESS */
      if( (uc&BIT7MASK) == (icVal&BIT7MASK) )        /* DQ7 == DATA  */
         return FLASH_SUCCESS;

      /* Step 6: Else return FLASH_POLL_FAIL */
      else                                   /* DQ7 here means fail */
         return FLASH_POLL_FAIL;
   }
}



void FlashBurnFlash(char *cRomImage,int iRomSize, char *cWhere)
{ 
    int iResult;
    int iLoop;
    int iBlockCount=0;
    unsigned char cBlocksToErase[MAXBLOCKOFFSETCONSTS]; 
//    unsigned int iDeviceCode;   /* Holds the return value */
    unsigned long *l_FlashMapToUse;
    unsigned long iNumBlocks;
    unsigned long aHV;
    
    iRomSize++;
    iRomSize >>= 1;
    aHV = (UL)cWhere - (UL)BASE_ADDR;     // damit erster Sektor bei 0 anfaengt
    aHV >>= 1;
    cWhere = (char*)aHV;
            // hardcode to 128N Flash
    iNumBlocks = NUM_BLOCKS_128N;
    l_FlashMapToUse = (unsigned long *)BlockOffsetConst128N;

    /* Return to Read Array mode */
    FlashWrite( 0x0000L, FLASH_PATTERN_ENTER_READ_MODE ); /* Use single instruction cycle method */


/* WEIRD!!! The relocation pragma also loads the relative address of the flash map!!!
    BlockOffsetConstAdjusted = ((unsigned char *)l_FlashMapToUse)+(int)cOffset; */
    /* RAc -- we don't want to erase the entire chip, only the range covered by the parameters cWhere and iRomSize!*/ 
/*    iResult=FLASHCHIPERASE(" move.l `cResults`,-(A7)"); */

    /* Finde erst den ersten Block heraus, in dem der Bereich liegt */
    for (iLoop=0;iLoop<iNumBlocks-1;iLoop++)
        if ((l_FlashMapToUse[iLoop]<=(unsigned long)cWhere) && (l_FlashMapToUse[iLoop+1]>(unsigned long)cWhere))
            cBlocksToErase[iBlockCount++]=(unsigned char)iLoop;
    if (!iBlockCount)  /* nur der letzte Block wird gebraucht */
        cBlocksToErase[iBlockCount++]=iNumBlocks-1;
    else  /* wir mssen gucken, ob mehr als ein Block gebraucht wird */
    {
        int iEndAddress = (int)cWhere+iRomSize;
        for (iLoop=(int)cBlocksToErase[0]+1;iLoop<iNumBlocks;iLoop++)
            if (iEndAddress>=l_FlashMapToUse[iLoop])
                cBlocksToErase[iBlockCount++]=(unsigned char)iLoop;
    }
    iResult=FlashBlockErase(iBlockCount,cBlocksToErase,l_FlashMapToUse);
    if (iResult!=FLASH_SUCCESS)
        return;
    FlashProgram((UL)cWhere,iRomSize,cRomImage);
}


#pragma section Flashcode end

